﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;


/* Query clauses and keywords */
    
namespace Lessons
{
    public partial class MainForm : Form
    {
        public MainForm()
        {
            InitializeComponent();
        }


        class Country
        {
            public string Code;
            public string Name;
            public List<string> Cities;
        }

        class Customer
        {
            public string Name;
            public string Country;
            public double Debit;
        }

        List<Country> countries = new List<Country>
        {
            new Country {Code = "ES", Name = "Spain", Cities = new List<string>{"Madrid", "Barcelona", "Valencia"} },
            new Country {Code = "CZ", Name = "Czech Republic", Cities = new List<string>{"Prague", "Brno"} },
            new Country {Code = "GR", Name = "Greece", Cities = new List<string>{"Athens", "Thessaloniki"} },
            new Country {Code = "DK", Name = "Denmark", Cities = new List<string>{"Copenhagen", "Aarhus"} }, 
            new Country {Code = "UK", Name = "United Kingdom", Cities = new List<string>{"London", "Edinburgh"} }, 
        };

        List<Customer> customers = new List<Customer>
        {
            new Customer {Name = "Cheap Software Ltd", Country = "GR", Debit = 35.6},
            new Customer {Name = "Fast Internet Co", Country = "DK", Debit = 51.7},
            new Customer {Name = "Neat Hardware", Country = "ES", Debit = 12.1},
            new Customer {Name = "Cool Tech", Country = "DK", Debit = 23.5},
            new Customer {Name = "Steady Logic", Country = "CZ", Debit = 49.4},
            new Customer {Name = "Nasty Data", Country = "GR", Debit = 62.2}  
        };




        /* sub-query with a compound from clause in order to access an inner list 
         
           q is IEnumerable<string> */
        private void button1_Click(object sender, EventArgs e)
        {
            var q = from country in countries
                        from city in country.Cities
                        where city.StartsWith("A") || city.StartsWith("B")
                        orderby city descending
                        select city;

            string S = "";

            foreach (string city in q)
                S += city + Environment.NewLine;

            MessageBox.Show(S);            

        }



        /* sub-query based on a result list coming from the first query 
           Each from clause can be thought of as a separated foreach statement 
         
           q is IEnumerable<Customer> */
        private void button2_Click(object sender, EventArgs e)
        {
            var q = from country in countries
                    where (country.Code == "GR") || (country.Code == "DK")
                        from customer in customers
                        where (customer.Country == country.Code) && (customer.Debit > 40)
                        orderby customer.Name
                        select customer;


            string S = "";

            foreach (Customer c in q)
                S += c.Name + " (" + c.Country + "): " + c.Debit.ToString() + Environment.NewLine;

            MessageBox.Show(S);
        }



        /* sub-query.
           the above example split into two steps and uses two distinct query variables.
           The second query uses the IEnumerable<T>.Contains() extension method          
         
           q is IEnumerable<string>
           q2 is IEnumerable<Customer>
           */
        private void button3_Click(object sender, EventArgs e)
        {
            var q = from country in countries
                    where (country.Code == "GR") || (country.Code == "DK")
                    select country.Code;

            var q2 = from customer in customers
                     where q.Contains(customer.Country) && (customer.Debit > 40)
                     select customer;


            string S = "";

            foreach (Customer c in q2)
                S += c.Name + " (" + c.Country + "): " + c.Debit.ToString() + Environment.NewLine;

            MessageBox.Show(S);
        }



        /* join 
           
           A LINQ join joins two sequences in an equii-join based on some equality condition.
           Joins in LINQ are always inner (equijoins) joins.
         
           q is IEnumerable<a> where a is an anonymous type of new { string Name, string Country }
           */
        private void button4_Click(object sender, EventArgs e)
        {
            var q = from customer in customers
                    join country in countries on customer.Country equals country.Code
                    orderby country.Code
                    select new { Name = customer.Name, Country = country.Name };

            string S = "";

            foreach (var c in q)
                S += c.Name + " (" + c.Country + ")" + Environment.NewLine;

            MessageBox.Show(S);
        }



        /* join 
           
           A LINQ join joins two sequences in an equii-join based on some equality condition.
           Joins in LINQ are always inner (equijoins) joins.
 
           In this version the result is the same as above, although datasources
           are used here in reverse order 
         
           q is IEnumerable<a> where a is an anonymous type of new { string Name, string Country }  */
        private void button5_Click(object sender, EventArgs e)
        {
            var q = from country in countries
                    join customer in customers on country.Code equals customer.Country
                    orderby country.Code
                    select new { Name = customer.Name, Country = country.Name };

            string S = "";

            foreach (var c in q)
                S += c.Name + " (" + c.Country + ")" + Environment.NewLine;

            MessageBox.Show(S);
        }



        /*  group join. 
            
            A LINQ join joins two sequences in an equii-join based on some equality condition.
            Joins in LINQ are always inner (equijoins) joins.
           
            The use of keyword into has a special effect when used with LINQ joins.

            For each element of the left sequence it creates an IEnumerable<T> list, where T 
            is the element type of the right sequence. That list contains those elements of the 
            right sequence which pass the equality condition. This is grouping actually.

            So joins using the keyword into are called group joins.
            There is no SQL equivalent for LINQ group join.     

            In general the keyword into is used to create an identifier (which is another implicitly typed local variable)
            which refers to the results of a group, join or select clause. 
            That identifier can then be used further as a datasource for additional query commands.

            customersPerCountry is IEnumerable<Customer>
            q is IEnumerable<a> where a is an anonymous type of new { string Country, IEnumerable<Customer> Customers } 
         
         */
        private void button6_Click(object sender, EventArgs e)
        {
            var q = from country in countries
                    join customer in customers on country.Code equals customer.Country into customersPerCountry
                    orderby country.Code
                    select new { Country = country.Name, Customers = customersPerCountry };


            string S = "";

            // it takes two foreach statements to iterate through the group join results.
            // The at variable stands for anonymous type
            foreach (var at in q)
            {
                S += at.Country + Environment.NewLine;
                foreach (Customer customer in at.Customers)
                    S += "     " + customer.Name + Environment.NewLine;
            }

            MessageBox.Show(S);
        }



        /* join
           simulating a left outer join 
         
           The left join simulation is done by using a second iteration (from clause)
           over the sequence produced by the group join and referenced by the keyword into,
           and forcing ALL the elements of that produced sequence into the final sequence, 
           by using the DefaultIfEmpty() extension method .  
        
           customersPerCountry is IEnumerable<Customer>
           q is IEnumerable<a> where a is an anonymous type of new { string Country, string Customer } 
         */
        private void button7_Click(object sender, EventArgs e)
        {

            var q = from country in countries
                    join customer in customers on country.Code equals customer.Country into customersPerCountry
                    orderby country.Code
                        from v in customersPerCountry.DefaultIfEmpty()
                        select new { Country = country.Name, Customer = (v == null ? string.Empty : v.Name) };

            string S = "";

            foreach (var c in q)
                S += "(" + c.Country + ") " + c.Customer + Environment.NewLine;

            MessageBox.Show(S);
        }



        /* group .. by clause 
           
          NOTE: a query body must end with a select or group..by clause 
         
          The result of a group..by clause is actually a 
                IEnumerable<IGrouping<TKey, TElement>>         
          that is a sequence of IGrouping<TKey, TElement> objects.
          Each IGrouping<TKey, TElement> object may contain zero or more elements that match the 
          key condition for the group.      
         
         q is a IEnumerable<IGrouping<string, Customer>>
         g is a IGrouping<string, Customer>
         */
        private void button8_Click(object sender, EventArgs e)
        {
            /* here, the implicitly typed variable q would be defined as 
               IEnumerable<IGrouping<string, Customer>> q = ...  */
            var q = from customer in customers
                    orderby customer.Country
                    group customer by customer.Country;

            string S = "";

            // it takes two foreach statements to iterate through the group results.
            // The g variable stands for group
            foreach (var g in q)
            {
                S += g.Key + Environment.NewLine;
                foreach (Customer customer in g)
                    S += "     " + customer.Name + Environment.NewLine;
            }

            MessageBox.Show(S);

        }




        /* another group..by example using an into clause.
           
           q is IOrderedEnumerable<IGrouping<char, Customer>>
           g is IGrouping<char, Customer> */
        private void button9_Click(object sender, EventArgs e)
        {
            var q = from customer in customers
                    group customer by customer.Name[0] into g
                    orderby g.Key
                    select g;


            string S = "";

            foreach (var g in q)
            {
                S += g.Key + Environment.NewLine;
                foreach (Customer customer in g)
                    S += "     " + customer.Name + Environment.NewLine;
            }

            MessageBox.Show(S);
       
        }



        /*  grouping and using aggregate functions (sum)            
                         
            A very frequent need is to group a datasource and then apply some aggregate function. 
            One way to achieve this is to group data using an into clause and then call the appropriate 
            aggregate extension method, possibly using a lambda expression.

            The cc is an IGrouping<string, Customer> 
            The q is an IEnumerable<a> where a is an anonymous type of new { string Country, double Debit } */
        private void button10_Click(object sender, EventArgs e)
        {
            var q = from customer in customers
                    group customer by customer.Country into cc
                    orderby cc.Key
                    select new { Country = cc.Key, Debit = cc.Sum(c => c.Debit) };
                    

            string S = "";

            foreach (var v in q)
                S += v.Country + ": " + v.Debit.ToString() + Environment.NewLine;

            MessageBox.Show(S);

        }

        /* the keyword let and using aggregate functions (sum)
         
           The let keyword can be used inside a query expression in order to create and initialize additional 
           range variables. Those range variables are considered as constants inside the expression. 
           It may be used in further queries though, if they store a queryable type.      
           
           DebitList is an IEnumerable<double> 
           q is an IEnumerable<a> where a is an anonymous type of new { string Country, double Debit } */
        private void button11_Click(object sender, EventArgs e)
        {
            var q = from country in countries
                    let DebitList = (
                        from customer in customers
                        where (customer.Country == country.Code)
                        select customer.Debit)
                    orderby country.Code
                    select new { Country = country.Code, Debit = DebitList.Sum() };


            string S = "";

            foreach (var v in q)
                S += v.Country + ": " + v.Debit.ToString() + Environment.NewLine;

            MessageBox.Show(S);
        }

    }
}
